/*
 * HugeFileCleaner.java 9 may 2016
 *
 * Copyright (c) 2016 Emmanuel PUYBARET / eTeks <info@eteks.com>. All Rights Reserved.
 */
package com.eteks.sweethome3d.utilities;

import java.awt.*;
import java.io.*;
import java.util.*;
import java.util.zip.*;
import javax.swing.*;

/**
 * Removes useless entries from a .sh3d file.
 * @author Emmanuel Puybaret
 */
public class HugeFileCleaner {
  public static void main(String [] args) {
    try {
      // Apply current system look and feel if swing.defaultlaf isn't defined
      UIManager.setLookAndFeel(System.getProperty("swing.defaultlaf", UIManager.getSystemLookAndFeelClassName()));
    } catch (Exception ex) {
      // Too bad, keep current look and feel
    }
    
    FileDialog fileDialog = new FileDialog(new Frame(), "Select SH3D file to fix", FileDialog.LOAD);
    fileDialog.setFilenameFilter(new FilenameFilter() {
        public boolean accept(File dir, String name) {
          return name.toLowerCase().endsWith(".sh3d");
        }
      });
    fileDialog.setVisible(true);
    if (fileDialog.getFile() != null) {
      File homeFile = new File(fileDialog.getDirectory(), fileDialog.getFile());
      Map<String, String> imageInSubFolderEntries = null;
      String savedFile = null;
      try {
        imageInSubFolderEntries = getImageInSubFolderEntries(homeFile);
        if (imageInSubFolderEntries == null) {
          JOptionPane.showMessageDialog(null, "Please save your file with Sweet Home 3D 4.4 or higher", "Missing ContentDigests", JOptionPane.ERROR_MESSAGE);
        } else if (imageInSubFolderEntries.isEmpty()) {
          JOptionPane.showMessageDialog(null, "File seems correct.", "File cleaner", JOptionPane.WARNING_MESSAGE);
        } else {
          fileDialog.setTitle("Select destination SH3D file");
          fileDialog.setMode(FileDialog.SAVE);
          fileDialog.setVisible(true);
          savedFile = fileDialog.getFile();
        }
      } catch (IOException ex) {
        JOptionPane.showMessageDialog(null, "Can't read file " + ex.getMessage(), "File cleaner", JOptionPane.ERROR_MESSAGE);
      }

      try {
        if (savedFile != null) {
          File destinationFile = new File(fileDialog.getDirectory(), savedFile);
          if (destinationFile.equals(homeFile)) {
            JOptionPane.showMessageDialog(null, "Destination file must be different.", "File cleaner", JOptionPane.ERROR_MESSAGE);
          } else {
            // Show a dialog with a progress bar
            JProgressBar progressBar = new JProgressBar();
            JOptionPane optionPane = new JOptionPane(new JComponent [] {new JLabel("Writing cleaned file..."), progressBar});
            optionPane.setOptions(new Object [0]);
            JDialog dialog = optionPane.createDialog(null, "File cleaner");
            dialog.setModal(false);
            dialog.setVisible(true);
            copyFileExceptSubFolderEntries(homeFile, destinationFile, imageInSubFolderEntries, progressBar.getModel());
            dialog.setVisible(false);
            JOptionPane.showMessageDialog(null, "File fixed!", "File cleaner", JOptionPane.INFORMATION_MESSAGE);
          }
        }
      } catch (IOException ex) {
        JOptionPane.showMessageDialog(null, "Can't save file " + ex.getMessage(), "File cleaner", JOptionPane.ERROR_MESSAGE);
      }
    }
    System.exit(0);
  }

  /**
   * Returns the entries in sub folders of the given home file that that reference an image.
   */
  private static Map<String, String> getImageInSubFolderEntries(File homeFile) throws IOException {
    ZipFile zipIn = null;
    try {
      // Read the content of the entry named "ContentDigests" if it exists
      zipIn = new ZipFile(homeFile);
      ZipEntry contentDigestsEntry = zipIn.getEntry("ContentDigests");
      if (contentDigestsEntry == null) {
        return null;
      } else {
        Map<String, String> imageInSubFolderEntries = new HashMap<String, String>();
        BufferedReader reader = new BufferedReader(new InputStreamReader(
            zipIn.getInputStream(contentDigestsEntry), "UTF-8"));
        String line = reader.readLine();
        if (line != null
            && line.trim().startsWith("ContentDigests-Version: 1")) {
          // Read Name lines that store an image in a subfolder
          while ((line = reader.readLine()) != null) {
            if (line.startsWith("Name:")) {
              String entryName = line.substring("Name:".length()).trim();
              String entryNameLowerCase = entryName.toLowerCase();
              if ((entryNameLowerCase.endsWith(".jpg") || entryNameLowerCase.endsWith(".png") || entryNameLowerCase.endsWith(".bmp")) 
                  && entryName.indexOf('/') > 0 
                  && entryName.lastIndexOf('/') != entryName.indexOf('/')) {
                imageInSubFolderEntries.put(entryName.substring(0, entryName.indexOf('/')), entryName);
              }
            } 
          }
        }
        return imageInSubFolderEntries;
      }
    } finally {
      if (zipIn != null) {
        zipIn.close();
      }
    }
  }

  /**
   * Copies all the entries of the given home file in the destination file, except 
   * <code>ContentDigests</code> entry and the siblings entries of values in <code>imageInSubFolderEntries</code>.
   */
  private static void copyFileExceptSubFolderEntries(File homeFile, File destinationFile,
                                                     Map<String, String> imageInSubFolderEntries, 
                                                     BoundedRangeModel rangeModel) throws IOException {
    ZipFile zipfile = null;
    try {
      // Get entries count in zip file
      zipfile = new ZipFile(homeFile);
      rangeModel.setMaximum(zipfile.size());
    } finally {
      if (zipfile != null) {
        zipfile.close();
      }
    }

    ZipInputStream zipIn = null;
    ZipOutputStream zipOut = null;
    try {
      zipOut = new ZipOutputStream(new FileOutputStream(destinationFile));
      zipOut.setLevel(9);
      zipIn = new ZipInputStream(new FileInputStream(homeFile));
      for (ZipEntry zipEntry = null; (zipEntry = zipIn.getNextEntry()) != null; ) {
        String entryName = zipEntry.getName();
        if (!entryName.equals("ContentDigests")) {
          String entryFolder = entryName.indexOf('/') > 0 
              ? entryName.substring(0, entryName.indexOf('/'))
              : null;
          if (entryFolder == null
              || imageInSubFolderEntries.get(entryFolder) == null
              || imageInSubFolderEntries.get(entryFolder).equals(entryName)) {
            writeZipEntry(zipOut, zipIn, entryName);
          }
        }
        // Update progression
        rangeModel.setValue(rangeModel.getValue() + 1);
      }
    } finally {
      if (zipOut != null) {
        zipOut.close();
      }
      if (zipIn != null) {
        zipIn.close();
      }
    }
  }
  
  /**
   * Writes in <code>zipOut</code> stream a new entry named <code>entryName</code> that 
   * match the data of the current entry in <code>zipIn</code>.
   */
  private static void writeZipEntry(ZipOutputStream zipOut, ZipInputStream zipIn, String entryName) throws IOException {
    byte [] buffer = new byte [8192];
    zipOut.putNextEntry(new ZipEntry(entryName));
    int size; 
    while ((size = zipIn.read(buffer)) != -1) {
      zipOut.write(buffer, 0, size);
    }
    zipOut.closeEntry();  
  }
}

